#include "StdAfx.h"
#include ".\Log.h"
#include ".\scriptHeaders.h"
#include ".\FunctionInstance.h"
#include ".\HashTable.h"
#include ".\opcodes.h"
#include ".\buffer.h"
#include <stdlib.h>
#include <stdio.h>
#include ".\list.h"
#include ".\Interfaces.h"
#include ".\script.h"
#include ".\virtualmachine.h"

//*******************************************************************************************************************
//Konstruktor
//*******************************************************************************************************************
CVirtualMachine::CVirtualMachine(void)
{
	//inicilizace pole s ukazateli na obsluzne funkce
	m_Functions = new EXEC_FUNC[NUM_OPCODES];

	//naplneni tabulky, do prislusnych polozek ulozime ukazatele na prislusne funkce
	m_Functions[nopCI] = (EXEC_FUNC)Exec_nop;
	m_Functions[mulCI] = (EXEC_FUNC)Exec_mul;
	m_Functions[negCI] = (EXEC_FUNC)Exec_neg;
	m_Functions[modCI] = (EXEC_FUNC)Exec_mod;
	m_Functions[subCI] = (EXEC_FUNC)Exec_sub;
	m_Functions[divCI] = (EXEC_FUNC)Exec_div;
	m_Functions[addCI] = (EXEC_FUNC)Exec_add;
	m_Functions[lgotoCI] = (EXEC_FUNC)Exec_goto;
	m_Functions[ifeqCI] = (EXEC_FUNC)Exec_ifeq;
	m_Functions[ifneCI] = (EXEC_FUNC)Exec_ifne;
	m_Functions[if_cmpeqCI] = (EXEC_FUNC)Exec_if_cmpeq;
	m_Functions[if_cmpgtCI] = (EXEC_FUNC)Exec_if_cmpgt;
	m_Functions[if_cmpltCI] = (EXEC_FUNC)Exec_if_cmplt;
	m_Functions[if_cmpleCI] = (EXEC_FUNC)Exec_if_cmple;
	m_Functions[if_cmpgeCI] = (EXEC_FUNC)Exec_if_cmpge;
	m_Functions[if_cmpneCI] = (EXEC_FUNC)Exec_if_cmpne;
	m_Functions[nreturnCI] = (EXEC_FUNC)Exec_return;
	m_Functions[vreturnCI] = (EXEC_FUNC)Exec_vreturn;
	m_Functions[loadCI] = (EXEC_FUNC)Exec_load;
	m_Functions[storeCI] = (EXEC_FUNC)Exec_store;
	m_Functions[ldc_intCI] = (EXEC_FUNC)Exec_ldc_int;
	m_Functions[ldc_stringCI] = (EXEC_FUNC)Exec_ldc_string;
	m_Functions[ldc_doubleCI] = (EXEC_FUNC)Exec_ldc_double;
	m_Functions[dupCI] = (EXEC_FUNC)Exec_dup;
	m_Functions[popCI] = (EXEC_FUNC)Exec_pop;
	m_Functions[lcallCI] = (EXEC_FUNC)Exec_lcall;
	m_Functions[ecallCI] = (EXEC_FUNC)Exec_ecall;
	m_Functions[shlCI] = (EXEC_FUNC)Exec_shl;
	m_Functions[shrCI] = (EXEC_FUNC)Exec_shr;
	m_Functions[incCI] = (EXEC_FUNC)Exec_inc;
	m_Functions[decCI] = (EXEC_FUNC)Exec_dec;
	m_Functions[andCI] = (EXEC_FUNC)Exec_and;
	m_Functions[orCI] = (EXEC_FUNC)Exec_or;
}

//*******************************************************************************************************************
//Destruktor
//*******************************************************************************************************************
CVirtualMachine::~CVirtualMachine(void)
{
}

//*******************************************************************************************************************
//Souisi s funkci rozhrani
//*******************************************************************************************************************
HRESULT CVirtualMachine::AddRef()
{
	return m_dwRef++;
}
//*******************************************************************************************************************
//Souvisi s funkci rozhrani
//*******************************************************************************************************************
HRESULT CVirtualMachine::Release()
{
	if (m_dwRef > 0)
		m_dwRef--;

	if (m_dwRef == 0)
	{
		delete this;
		return 0;
	}
	else
		return m_dwRef;
}
//*******************************************************************************************************************
//instrukce nop
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_nop(CFunctionInstance* func,ScriptDependencies* sd)
{
	return 1;
}
//*******************************************************************************************************************
//instrukce mul
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_mul(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi jejich nasobek
	StackItem var_right,var_left;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	func->PushStackItem(var_left * var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce neg
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_neg(CFunctionInstance* func,ScriptDependencies* sd)
{
	//neguje promennou na vrcholu zasobniku
	StackItem var,var_result;

	var = func->PopStackItem();

	//podivame se o jakou promennou jde a podle toho se zaridime
	switch (var.kind)
	{
	case INT_CONST:
		var_result.kind = INT_CONST;	//integer, bez problemu
		var_result.intval = -(var.intval);
		break;
	case DOUBLE_CONST:
		var_result.kind = DOUBLE_CONST;	//double, bez problemu
		var_result.doubleval = -(var.doubleval);
		break;
	case ZSTRING_CONST:
		//retezec, preved na double. nevime jestli v nem je int, nebo double nebo neco uplne jineho
		//prevedeme tedy na double a bud to bude cislo nebo 0
		var.Cast_stringtodouble();
		var_result.kind = DOUBLE_CONST;
        var_result.doubleval = -(var.doubleval);
		break;
	}

	func->PushStackItem(var_result);

	return 1;
}
//*******************************************************************************************************************
//instrukce mod
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_mod(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi zbytek po celociselnem deleni
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	func->PushStackItem(var_left % var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce sub
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_sub(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi jejich rozdil
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	func->PushStackItem(var_left - var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce div
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_div(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi jejich podil
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	//vyuziva pretizeny operator / tridy StackItem
	func->PushStackItem(var_left / var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce add
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_add(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi jejich soucet
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();
	
	func->PushStackItem(var_left + var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce shl
//*******************************************************************************************************************
int CVirtualMachine::Exec_shl(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi levy operand posunuty o pravy operand bitu vlevo
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();
	
	func->PushStackItem(var_left << var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce shr
//*******************************************************************************************************************
int CVirtualMachine::Exec_shr(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi levy operand posunuty o pravy operand bitu vpravo
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();
	
	func->PushStackItem(var_left >> var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce and
//*******************************************************************************************************************
int CVirtualMachine::Exec_and(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi logicky soucin operandu
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();
	
	func->PushStackItem(var_left & var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce or
//*******************************************************************************************************************
int CVirtualMachine::Exec_or(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme ze zasobniku dva operandy a na vrchol ulozi logicky soucet operandu
	StackItem var_left,var_right;

	var_right = func->PopStackItem();
	var_left = func->PopStackItem();
	
	func->PushStackItem(var_left | var_right);

	return 1;
}
//*******************************************************************************************************************
//instrukce inc
//*******************************************************************************************************************
int CVirtualMachine::Exec_inc(CFunctionInstance* func,ScriptDependencies* sd)
{
	//inkrementuje promennou na vrcholu zasobniku
	StackItem var;

	var = func->PopStackItem();

	var++;

	func->PushStackItem(var);

	return 1;
}
//*******************************************************************************************************************
//instrukce dec
//*******************************************************************************************************************
int CVirtualMachine::Exec_dec(CFunctionInstance* func,ScriptDependencies* sd)
{
	//dekrementuje promennou na vrcholu zasobniku
	StackItem var;

	var = func->PopStackItem();

	var--;

	func->PushStackItem(var);

	return 1;
}
//*******************************************************************************************************************
//instrukce goto
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_goto(CFunctionInstance* func,ScriptDependencies* sd)
{
	//uskutecni nepodmineny skok
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->SetPC(adress);
	return 1;
}
//*******************************************************************************************************************
//instrukce ifeq
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_ifeq(CFunctionInstance* func,ScriptDependencies* sd)
{
	//skoc pokud integer na vrcholu zasobniku je nula
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var;

	var = func->PopStackItem();
	switch (var.kind)
	{
	case INT_CONST:
		break;
	case DOUBLE_CONST:
		var.Cast_doubletoint();
		break;
	case ZSTRING_CONST:
		var.Cast_stringtoint();
		break;
	}

	if (var.intval == 0)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce ifne
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_ifne(CFunctionInstance* func,ScriptDependencies* sd)
{
	//skoc pokud integer na vrcholu zasobniku neni 0
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var;

	var = func->PopStackItem();
	switch (var.kind)
	{
	case INT_CONST:
		break;
	case DOUBLE_CONST:
		var.Cast_doubletoint();
		break;
	case ZSTRING_CONST:
		var.Cast_stringtoint();
		break;
	}

	if (var.intval != 0)
		func->SetPC(adress);
	return 1;
}
//*******************************************************************************************************************
//instrukce if_cmpeq
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_if_cmpeq(CFunctionInstance* func,ScriptDependencies* sd)
{
	//podmineny skok, pokud dva operandy na vrcholu zasobniku se rovnaji
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var_left,var_right;
	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	if (var_left == var_right)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce if_cmpgt
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_if_cmpgt(CFunctionInstance* func,ScriptDependencies* sd)
{
	//podmineny skok, pokud levy operand je vetsi nez pravy
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var_left,var_right;
	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	if (var_left > var_right)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce if_cmplt
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_if_cmplt(CFunctionInstance* func,ScriptDependencies* sd)
{
	//podmineny skok, pokud levy operand je mensi nez pravy
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var_left,var_right;
	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	if (var_left < var_right)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce if_cmple
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_if_cmple(CFunctionInstance* func,ScriptDependencies* sd)
{
	//podmineny skok, pokud levy operand je mensi nebo roven pravemu
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var_left,var_right;
	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	if (var_left <= var_right)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce if_cmpge
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_if_cmpge(CFunctionInstance* func,ScriptDependencies* sd)
{
	//podmineny skok, pokud levy operand je vetsi nebo roven pravemu
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var_left,var_right;
	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	if (var_left >= var_right)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce if_cmpne
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_if_cmpne(CFunctionInstance* func,ScriptDependencies* sd)
{
	//podmineny skok, pokud operandy na vrcholu zasobniku se nerovnaji
	unsigned int adress = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	StackItem var_left,var_right;
	var_right = func->PopStackItem();
	var_left = func->PopStackItem();

	if (var_left != var_right)
		func->SetPC(adress);

	return 1;
}
//*******************************************************************************************************************
//instrukce return
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_return(CFunctionInstance* func,ScriptDependencies* sd)
{
	//provede navrat z funkce a nevraci se hodnota
	int arg = func->GetNumberOfParams();

	func->PopActivation();

	func->Pop(arg);
	return 1;
}
//*******************************************************************************************************************
//instrukce load
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_load(CFunctionInstance* func,ScriptDependencies* sd)
{
	//nacte promennou na vrchol zasobniku
	unsigned short adress = *((unsigned short*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned short));

	//zeptame se na typ promenne co je v zasobniku na adrese adress
	int var_type = func->GetLocalVariableKind(adress);

	switch (var_type)
	{
	case INT_CONST:
		func->PushInt(func->GetLocalInt(adress));
		break;
	case DOUBLE_CONST:
		func->PushDouble(func->GetLocalDouble(adress));
		break;
	case ZSTRING_CONST:
		func->PushString(func->GetLocalString(adress));
		break;
	}

	return 1;
}
//*******************************************************************************************************************
//instrukce store
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_store(CFunctionInstance* func,ScriptDependencies* sd)
{
	//ulozi vrchol zasobniku do promenne
	unsigned short adress = *((unsigned short*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned short));

	//zeptej se na typ promenne co je na vrcholu zasobniku
	int var_type = func->GetStackVariableKind();

	switch (var_type)
	{
	case INT_CONST:
		func->SetLocalInt(func->PopInt(),adress);
		break;
	case DOUBLE_CONST:
		func->SetLocalDouble(func->PopDouble(),adress);
		break;
	case ZSTRING_CONST:
		func->SetLocalString(func->PopString(),adress);
		break;
	}

	return 1;
}
//*******************************************************************************************************************
//instrukce ldc_int
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_ldc_int(CFunctionInstance* func,ScriptDependencies* sd)
{
	//nacte integer z tabulky konstant na vrchol zasobniku
	unsigned int index = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));
	
	func->PushInt(sd->m_ConstantTable[index].intval);
	return 1;
}
//*******************************************************************************************************************
//instrukce ldc_string
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_ldc_string(CFunctionInstance* func,ScriptDependencies* sd)
{
	//nacte retezec z tabulky konstant na vrchol zasobniku
	unsigned int index = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	func->PushString(sd->m_ConstantTable[index].strval);
	return 1;
}
//*******************************************************************************************************************
//instrukce ldc_double
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_ldc_double(CFunctionInstance* func,ScriptDependencies* sd)
{
	//nacte double z tabulky konstant na vrchol zasobniku
	unsigned int index = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	func->PushDouble(sd->m_ConstantTable[index].doubleval);

	return 1;
}
//*******************************************************************************************************************
//instrukce dup
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_dup(CFunctionInstance* func,ScriptDependencies* sd)
{
	//vezme vrchol zasobniku a ulozi ho do zasobniku este jednou
	func->PushStackItem(func->GetStackItem());
	return 1;
}
//*******************************************************************************************************************
//instrukce pop
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_pop(CFunctionInstance* func,ScriptDependencies* sd)
{
	//snizi vrchol zasobniku o 1
	func->Pop();
	return 1;
}
//*******************************************************************************************************************
//instrukce lcall
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_lcall(CFunctionInstance* func,ScriptDependencies* sd)
{
	//provede skok na lokalni funkci, funkci ktera je napsana ve skriptu
	unsigned int index = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));

	FUNCTION_HEADER* function = NULL;		//ukazatel na strukturu s informacemi o funkci
	//precteni jmena funkce z tabulky konstant
	bool found = sd->m_FunctionList->Get(sd->m_ConstantTable[index].strval, &function);

	//pokud bylo jmeno funkce nalezeno v tabulce konstant, musi platit promenna found a
	//ukazatel na funkci musi byt tez platny
	if (found && function != NULL)
	{
		//musime ulozit aktualni stack-pointer, ukazatel vrcholu zasobniku
		int iCurSP = func->GetSP();

		//zkontrolujeme jestli vrchol zasobniku + limit volane funkce neprekroci velikost zasobniku, pokud ano,
		//nahlasime do logu chybu a funkci volat nebudeme. Navic nastavime ukazatel kodu na konec programu, protoze
		//by nasledujici operace bez provedeni volane funkce nejspis nebyly spravne
		int stackHeight = (iCurSP - func->function->inputs) + func->function->localsLimit + func->function->stackLimit;
		if (stackHeight > MAX_STACK_SIZE)
		{
			Log("Error: Cannot call function '%s'. Not enought stack space to execute it! Terminating<br>",func->function->name);
			//ted opatreni aby fce skoncila
			func->SetPC(func->GetEndCode());
			return 0;
		}

		//ulozime na zasobnik volani funcki hodnoty nove funkce, tim vznikne novy vrchol zasobniku a stary sav zustane
		//ulozen. Budeme ho moci obnovit a pokracovat v provadeni kodu tam kde jsme skoncili. Jen nejdriv zkontrolujeme
		//jestli zasobnik volani funkci taky neni preplnen
		if (func->PushActivation() == ERR_STACKOVERFLOW)
		{
			Log("Error - Stack Overflow. Terminating execution...<br>");
			func->SetPC(func->GetEndCode());
			return 0;
		}

		//nastavim program counter-ukazatel kodu na novou adresu
		func->SetPC(function->startCode);
		//stejne tak nastavime konec kodu
		func->SetEndCode(function->endCode);

		//Tyto dva rdaky souvisi s nastavenim ukazatelu do zasobniku pro novou funkci. Pokud je vam jasne jak
		//pracuje zasobnik u naseho virtualniho stroje, rozlustite i tyto vypocty. Promenne funkci jsou pocitany
		//od argumentu funkce. Promenne jsou ulozeny v zasobniku od adresy kam ukazuje BSP. Hned po posledni promenne
		//je v zasobniku misto pro provadeni vypoctu a ukladani vysledku. Na vrchol tohoto zasobniku ukazuje SP.
		//Pri volani nove funkce musi tedy BSP ukazovat na misto kde je ulozena prvni promenna teto funkce a SP bude
		//ukazovat na zacatek zasobniku pro provadeni vypoctu. Bude tedy ukazovat na posledni promennou funkce. Protoze
		//pri prvnim pouziti nejake instrukce pro nacteni operandu do zasobniku bude nejprve SP inkrementovano, cimz se ukazatel
		//dostane na prvni misto za posledni promennou funkce
		func->SetBSP(iCurSP - (function->inputs - 1));
		func->SetSP(iCurSP - function->inputs + function->localsLimit);

		func->SetNumberOfParams(function->inputs);
	}
	else
		//tohle by se prakticky nemelo nikdy stat
		Log("ERROR - Function '%s' was not found in VM<br>",sd->m_ConstantTable[index].strval);

	return 1;
}
//*******************************************************************************************************************
//instrukce ecall
//*******************************************************************************************************************
int CALLBACK CVirtualMachine::Exec_ecall(CFunctionInstance* func,ScriptDependencies* sd)
{
	//zavola externi funkci
	unsigned int index = *((unsigned int*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned int));
	unsigned short params = *((unsigned short*)&(sd->m_CodeSegment[func->GetPC()]));
	func->AddPC(sizeof(unsigned short));

	ExternFunction*	pItem = NULL;	//ukazatel na strukturu s informacemi o funkci
	bool	found = sd->m_Imports->Get(sd->m_ConstantTable[index].strval,&pItem);

	//stejne jako u lokalni funkce, funkce musi byt nalezena v tabulce  a ukzatel na strukturu musi byt platny
	if (found && pItem)
	{
		//zkontrolujeme handle knihovny a adresu, aby nahodou nebyly neplatne, pokud se tak stane,
		//nemuzeme volani provest
		if ((!pItem->hDll) || (!pItem->address))
		{
			Log("Cannot call function '%s'. Address or module handle is invalid. Adress is: %Xh, Module Handle is: %Xh<br>",sd->m_ConstantTable[index].strval,pItem->address,pItem->hDll);
			return 0;
		}

        //ukladani parametru do zasobniku		
		for (unsigned int i = 0; i < params; i++)
		{
			//jak jiste vime, pri standartnim volani funkce pres assembler se musi argumenty funkce
			//ukladat v opacnem poradi nez jsou zapsany. Protoze ale nas skriptovaci jazyk je uklada do
			//zasobniku poporade, v tom poradi jak je zapiseme, staci argumenty jen ze zasobniku vybirat a ukladat
			//do zasobniku pocitace. A mame i zajisteno ze budou ulozeny od posledniho po prvy
			StackItem va_arg = func->PopStackItem();
			switch (va_arg.kind)
			{
			case INT_CONST:
				__asm push va_arg.intval;
				break;
			case ZSTRING_CONST:
				__asm push va_arg.strval;
				break;
			case DOUBLE_CONST:
				//__asm push va_arg.doubleval;
				break;
			}
		}

		//ziskame adresu funkce
		FARPROC address = pItem->address;
		//ted volani fce
		__asm call address;

		StackItem va_return;
		//vraci hodnotu?
		switch (pItem->type)
		{
		case INT_CONST:
			__asm mov va_return.intval, eax;
            va_return.kind = INT_CONST;
			func->PushStackItem(va_return);
			break;
		case ZSTRING_CONST:
			__asm mov va_return.strval, eax;
			va_return.kind = ZSTRING_CONST;
			func->PushStackItem(va_return);
			break;
		case DOUBLE_CONST:
			/*__asm mov va_return.doubleval, eax;
			va_return.kind = DOUBLE_CONST;
			func->PushStackItem(va_return);*/
			break;
		default:
			/*va_return.kind = INT_CONST;
			va_return.intval = 0;
			func->PushStackItem(va_return);*/
			break;
		}
	}

	return 1;
}
//*******************************************************************************************************************
//instrukce vreturn
//*******************************************************************************************************************
int CVirtualMachine::Exec_vreturn(CFunctionInstance* func,ScriptDependencies* sd)
{
	//uloz si posledni hodnotu, tj tu co se vraci
	StackItem var_return = func->PopStackItem();

	//kolik mela funkce parametru
	int args = func->GetNumberOfParams();

	//vrat se do puvodni fce, tj tady se obnovi obsah PC, BSP a SP
	func->PopActivation();

	//odloz argumenty funkce
    func->Pop(args);

	//uloz na vrchol zasobniku vracenou hodnotu
    func->PushStackItem(var_return);

	return 1;
}
//*******************************************************************************************************************
//funkce pro provedeni instrukce
//*******************************************************************************************************************
void CVirtualMachine::Run(CFunctionInstance* function,int iMaxInstructions)
{
	int iExecutedInstructions = 0;

	//kontrola uakazatele na zdroje
	if (!m_ScriptDependencies)
		return;

	//ukazatel na zdroje muze byt platny, ale co dilci ukazatele na jednotlive dalsi zdroje
	if (!m_ScriptDependencies->m_CodeSegment || !m_ScriptDependencies->m_FunctionList)
		return;

	//pokud maximalni pocet instrukci byl zadan 0, coz znaci konej dokud neskonci funkce, nastav mnoztvi provedenych
	//instrukci na -1, cimz vlastne zajisti ze nikdy nedojde k nesplneni podminky cyklu ktera hlida pocet instrukci
	if (iMaxInstructions == 0)
		iExecutedInstructions = -1;

	//dekodovaci smycka
	while ((function->GetPC() < function->GetEndCode()) && (function->status != STATUS_DEAD) && (iExecutedInstructions < iMaxInstructions))
	{
		//precti aktualni bytovy kod instrukce
		unsigned char opCode = m_ScriptDependencies->m_CodeSegment[function->GetPC()];
		//zvyc citac instrukci o bajt cimz se posuneme na dalsi bytovy kod
		function->AddPC(sizeof(char));

		//koukni se jestli se nejedna o neznamy kod instrukce a pokud jo, nahlas to a kod jednoduse preskoc
		//tim ze budes pokracovat dal
		if (opCode < 0 || opCode > (NUM_OPCODES - 1))
		{
			Log("Error. Invalid opcode '%i' found in code segment<br>",opCode);
		}
		else
		{
			//kod instrukce je platny, ziskej ukazatel na funkci ktera instrukci emuluje
			//bytovy kod je indexem do pole ukazatelu na obsluzne funkce. Tento zpusob je 
			//zrejme nejrychlejsi zpusob dekodovani instrukci, zadne rozhodovaci bloky ktere
			//hrozne zdrzuji. Pracuje to jako preruseni
			EXEC_FUNC funcPointer = (EXEC_FUNC)m_Functions[opCode];
			iExecutedInstructions += funcPointer(function,m_ScriptDependencies);
		}

		//opet hlidej pripad ze se ma pracovat dokud funkce neskonci
		if (iMaxInstructions == 0)
			iExecutedInstructions = -1;
	}

	//pokud funkce skoncila, vypis o tom ypravu do logu
	if (function->GetPC() >= function->GetEndCode())
	{
		Log("Killing function '%s'. PC is: '%i'. End of code is: '%i'<br>",function->function->name,(unsigned int)function->GetPC(),(unsigned int)function->GetEndCode());
		//nastav status funkce na dead, tj. uz byla provedena a bude diky tomu odstranena ze seznamu funkci
		//ktere cekaji na zpracovani
		function->status = STATUS_DEAD;
	}
}
//*******************************************************************************************************************
//inicializace
//*******************************************************************************************************************
void CVirtualMachine::Init(ScriptDependencies* scriptdependencies)
{
	//jen ulozeni ukazatele na zdroje, ktery je nezbytny pro funkci virtualniho stroje
	m_ScriptDependencies = scriptdependencies;
}